package com.cav.taglibs.mumps.cewolf;

import java.io.Serializable;
import java.util.*;
import java.util.logging.*;
import java.util.regex.*;

import javax.servlet.jsp.JspTagException;

import org.jfree.data.category.*;

import com.cav.mserver.MumpsQuery;
import com.cav.taglibs.mumps.JSPMumpsSession;

import de.laures.cewolf.*;
import de.laures.cewolf.links.LinkGenerator;
import de.laures.cewolf.tooltips.ToolTipGenerator;

/**
 * A Cewolf data producer, which pulls data from M query.
 * 
 * @author Uri Schor
 */
public class MumpsChartCewolfDataProducer implements DatasetProducer,
		LinkGenerator, ToolTipGenerator, Serializable {

	/** Pattern for the query's first column title, that holds the X and Y
	 axis' labels */
	protected static Pattern AXIS_LABEL_PATTERN = Pattern
			.compile("1\\.(.*) > 2\\.(.*)");

	/**
	 * Static initialization of the logger for this class
	 */
	private static Logger logger = Logger.getLogger(MumpsChartCewolfDataProducer.class
			.getPackage().getName());

	/** The series names */
	protected String[] seriesNames;

	private MumpsQuery query;

	/**
	 * 
	 */
	public MumpsChartCewolfDataProducer() {
		super();		
		logger.fine("MumpsChartCewolfDataProducer Created");
	}

	/**
	 * Execute the M code to get the query result.
	 * @param portlet
	 * @param rundata
	 * @throws Exception
	 */
	protected synchronized MumpsQuery executeQuery(String mcode)
			throws Exception {
		MumpsQuery query = new MumpsQuery(mcode, new HashMap());
		query.execute();
		return query;
	}

	/**
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		if (query != null) {
			query.clear();
			query = null;
		}
		logger.fine(this + " finalized.");
	}

	/**
	 * Returns a link target for a special data item.
	 */
	public String generateLink(Object data, int series, Object category) {
		if (seriesNames == null || seriesNames.length < series) {
			return null;
		}
		return seriesNames[series];
	}

	/**
	 * @see org.jfree.chart.tooltips.CategoryToolTipGenerator#generateToolTip(CategoryDataset, int, int)
	 */
	public String generateToolTip(CategoryDataset arg0, int series, int arg2) {
		if (seriesNames == null || seriesNames.length < series) {
			return null;
		}
		return seriesNames[series];
	}


	/**
	 * Returns a unique ID for this DatasetProducer
	 */
	public String getProducerId() {
		return "ZJData";
	}

	/**
	 * This producer's data is invalidated after 5 seconds. By this method the
	 * producer can influence Cewolf's caching behavior the way it wants to.
	 */
	public boolean hasExpired(Map params, Date since) {
		logger.entering(getClass().getName(), "hasExpired");
		return true;
		//return (System.currentTimeMillis() - since.getTime()) > 5000;
	}

	/**
	 *  Produces the data from the M query.
	 */
	public Object produceDataset(Map params) throws DatasetProduceException {
		logger.entering(getClass().getName(), "produceDataset");
		
		synchronized (MumpsChartCewolfDataProducer.class) {
			
		DefaultCategoryDataset dataset = new DefaultCategoryDataset() {
				/**
				 * @see java.lang.Object#finalize()
				 */
				protected void finalize() throws Throwable {
					super.finalize();
					logger.fine(this + " finalized.");
				}
			};

			// Execute the M query
			String mcode = (String) params.get("mcode");
			if (mcode == null) {
				logger.info("The Mumps code query is null, "
						+ "hence not generating the result set.");
				return dataset;
			}
			MumpsQuery query = null;
			try {
				query = executeQuery(mcode);
			}
			catch (Exception e) {
				logger.log(Level.SEVERE, "Error executing M query", e);
				return dataset;
			}

			// Get the X and Y axis' labels. They are in the first column 
			String xAxisLabel = "";
			String yAxisLabel = "";
			String legendLabel = "";
			if (query.getColumnTitles() == null
					|| query.getColumnTitles().size() < 2) {
				logger.severe("Query did not return at least 2 columns: <"
						+ mcode + '>');
				return dataset;
			}
			String[] colTitles = (String[]) query.getColumnTitles().toArray(
					new String[] {});
			try {
				String firstCol = colTitles[0];
				Matcher matcher = AXIS_LABEL_PATTERN.matcher(firstCol);
				if (matcher.matches()) {
					legendLabel = matcher.group(2);
					xAxisLabel = matcher.group(1);
				}
				yAxisLabel = colTitles[colTitles.length - 1];
			}
			catch (Exception e) {
				logger
						.fine("Badly formatted 1st column title: "
								+ colTitles[0]);
			}

			seriesNames = new String[colTitles.length - 2];
			System.arraycopy(colTitles, 1, seriesNames, 0, seriesNames.length);

			// Get the column values
			int columnNum = query.getColumnTitles().size();
			Iterator rsIter = query.getResultSet().iterator();
			while (rsIter.hasNext()) {
				List rsRow = (List) rsIter.next();
				String series = (String) rsRow.get(0);
				for (int i = 1; i < columnNum - 1; i++) {
					String value = (String) rsRow.get(i);
					double doubleVal = 0;
					try {
						doubleVal = (Double.parseDouble(value) * 100) / 100;
					}
					catch (NumberFormatException e) {
						logger.severe("Value is not numeric: <" + value + '>');
					}
					dataset.addValue(doubleVal, series, colTitles[i]);
				}
			}
			return dataset;
		}
	}
}
