package com.cav.mserver.tests;

import java.io.*;
import java.net.*;

/**
 * Benchmark for Mumps TCP server
 * @author Uri Schor
 */
public class MumpsTCPServerBenchmark extends HttpMuxBenchmark {

	/**
	 * An input stream that wraps the Mumps' console standard output and is used
	 * to read from it until the prompt, exclusive.
	 * 
	 * @author Uri Schor
	 */
	protected static class MumpsConsoleInputStream extends InputStream {

		/** The current index within the prompt */
		private int index = 0;

		/** The wrapped input stream */
		private InputStream is;

		/** The prompt as int array for performance */
		private final char[] prompt = {'~', '~', '~', '>', ' '};

		/** Did we receive prompt by now */
		private boolean receivedPrompt = false;

		/** The current char from the prompt to retansmit, in case we read a
		 * prefix of the prompt and it turned out to be something else */
		private int retransmit = -1;

		/** The character we should retransmnit in case we read a prefix of the
		 * prompt and then it arrived and broke the prompt sequence */
		private int retransnitChar;

		/**
		 * Create a new input stream wrapper
		 * @param session The Mumps session
		 * @param in The original input stream (stdout of Mumps console)
		 */
		public MumpsConsoleInputStream(InputStream is) {
			this.is = is;
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#available()
		 */
		public int available() throws IOException {
			if (receivedPrompt) {
				return 0;
			}
			else {
				return is.available();
			}
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#close()
		 */
		public void close() throws IOException {
			// We do not close the actual stream, since it's reused
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		public boolean equals(Object obj) {
			return is.equals(obj);
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#hashCode()
		 */
		public int hashCode() {
			return is.hashCode();
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#mark(int)
		 */
		public void mark(int readlimit) {
			is.mark(readlimit);
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#markSupported()
		 */
		public boolean markSupported() {
			return is.markSupported();
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#read()
		 */
		public int read() throws IOException {
			try {
				// Check if the prompt was fully read
				if (receivedPrompt) {
					return -1;
				}

				// Check if we read a prefix of the promprt we should re-transmit
				if (retransmit > -1) {
					if (retransmit < index) {
						return prompt[retransmit++];
					}
					else if (retransmit == index) {
						index = 0;
						retransmit = -1;
						return retransnitChar;
					}
				}

				// Read the next character
				int c;
				while ((c = is.read()) == prompt[index]) {
					++index;
					if (index == prompt.length) {
						// Prompt fullt received
						receivedPrompt = true;
						return -1;
					}
				}
				if (index > 0) {
					// We had read a prefix of the prompt, but it turns out it's
					// not the prompt after all.
					retransmit = 1;
					retransnitChar = c;
					return prompt[0];
				}
				else {
					return c;
				}
			}
			catch (IOException e) {
				throw e;
			}
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#reset()
		 */
		public void reset() throws IOException {
			is.reset();
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#skip(long)
		 */
		public long skip(long n) throws IOException {
			return is.skip(n);
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#toString()
		 */
		public String toString() {
			return is.toString();
		}

	}
	
	private static class ClientThread extends Thread {
		int nRequests;
		String host;
		int port;

		public ClientThread(String host, int port, int nRequests) {
			this.host = host;
			this.port = port;
			this.nRequests = nRequests;
		}
		
		/* (non-Javadoc)
		 * @see java.lang.Thread#run()
		 */
		public void run() {
			Socket socket = null;
			InputStream is = null;
			BufferedReader reader = null;
			PrintStream out = null;
			try {		
				socket = new Socket(host, port);
				socket.setTcpNoDelay(true);
				is = socket.getInputStream();
				reader = new BufferedReader(new InputStreamReader(is));
				out = new PrintStream(socket.getOutputStream());
			}
			catch (IOException e) {
				e.printStackTrace();
				return;
			}
			long start = System.currentTimeMillis();
			long writes = 0;
			long reads = 0;
			for (int i = 0; i < nRequests; ++i) {
				//sbuf.setLength(0);
				try {
					long before = System.currentTimeMillis();
					out.println("RTN=D ^JPOS&TYPE=901");
					out.flush();
					writes += (System.currentTimeMillis() - before);
					before = System.currentTimeMillis();
//					int c;
//					InputStream cis = new MumpsConsoleInputStream(is);
//					while ((c = cis.read()) != -1) {
//						//System.out.write(c);
//					}
					String line;
					while ((line = reader.readLine()) != null) {
						if (line.startsWith("~~~>")) {
							break;
						}
						//System.out.println(line);
					}
					reads += (System.currentTimeMillis() - before);
				}
				catch (IOException e1) {
					e1.printStackTrace();
					return;
				}
				//System.out.println("Got " + sbuf);
			}
			long duration = System.currentTimeMillis() - start;
			System.out.println("Took " + duration + " mSecs");
			System.out.println("Writes took " + writes + " mSecs");
			System.out.println("Reads took " + reads + " mSecs");
		}
	}


	public static void main(String[] args) {
		if (args.length != 4) {
			System.err.println("Usage: java "
								+ HttpMuxBenchmark.class.getName()
								+ " host port num_threads num_requests");
			System.exit(1);
		}

		String host = args[0];
		int port = Integer.parseInt(args[1]);;
		int nThreads = Integer.parseInt(args[2]);
		int nRequests = Integer.parseInt(args[3]);
		for (int i = 0; i < nThreads; ++i) {
			new ClientThread(host, port, nRequests).start();
		}
	}
}
