package com.cav.mserver;

import java.io.*;
import java.net.*;
import java.util.logging.*;
import java.util.logging.Logger;

public class TelnetClient {

	private static Logger logger = Logger.getLogger(TelnetClient.class
		.getPackage().getName());

    /* Notes whether we're akready connected */
    private boolean connected = false;

    private Socket socket;
    private InputStream inputStream;
    private OutputStream outputStream;
    private PrintWriter outWriter;
    
    public  static boolean sendEnterOnLogin = false;

    public  final static int OS_UNIX = 1;
    public  final static int OS_NT   = 2;

    public  final static int    TELNET_PORT = 23;

    public  final static String TELNET_DISCONNECTED = "TELNET SESSION DISCONNECTED!";

    private final static String USER_PROMPT_UNIX = "login: ";
    private final static String USER_PROMPT_NT = "Username: ";
    private final static String PASSWD_PROMPT = "Password:";
    private final static String CAV_JAVA_PROMPT = "~~~> " ;
    private final static String MENU_PROMPT = "dlert ywd" ;
    private final static String MENU_ENTRY_KEY = "CavJava" ;

    // the terminal type we pretend to be
    private final static String TERM = "dumb";

    /**
     * Create a new telnet client.
     * @param host The host name to connect to
     * @param port The port to connect to
     * @throws IOException
     */
    public TelnetClient(String host, int port) throws IOException {
    	this(host, port, 0);
    }
    
    /**
     * Create a new telnet client
     * @param host The host name to connect to
     * @param port The port to connect to
     * @param timeout The socket timeout on read() calls, in milliseconds. 
     * 0 (zero) tells to block indefinitely.
     * @throws IOException
     */
    public TelnetClient(String host, int port, int timeout) throws IOException {
        logger.info("Connecting to host: " + host + " port: " + port);
        
    	try {
    	    socket = new Socket();
            socket.setTcpNoDelay(true);
            socket.setSoTimeout(timeout);
    	    socket.connect(new InetSocketAddress(host, port), timeout);
    	}
        catch(ConnectException e) {
        	logger.log(Level.SEVERE, "Error connecting",e);
            throw e;
        }
    	catch(UnknownHostException e) {
    	    // just forward it
    	    throw e;
    	}
        catch(java.net.NoRouteToHostException e) {
            throw new IOException(TELNET_DISCONNECTED);
        }
        catch(IOException e) {
            // just forward it
            throw e;
        }
    

        inputStream = socket.getInputStream();
        outputStream = socket.getOutputStream();
        // create an auto-flushed (on 'println') PrintWriter
        outWriter = new PrintWriter(outputStream, true);

	    connected = true;
    }

    public void disconnect() throws Exception {
	if (! connected)
	    return;

	if (socket != null) {
	    socket.close();
	    socket = null;
	}
	inputStream = null;
	outputStream = null;
    }

    public java.io.InputStream getInputStream() {
	return inputStream;
    }

    public java.io.OutputStream getOutputStream() {
	return outputStream;
    }

    public boolean login(String user, String passwd) throws IOException {
        if (sendEnterOnLogin) {
            // By clicking just enter on the login, I get another prompt for login.
            // This is parametric (default true) since in Feldman there is a hang-up
            // when you enter a blank login.
            println("");                
        }

        int os;

        if (waitForString(USER_PROMPT_UNIX, true, USER_PROMPT_NT)){
            os = OS_UNIX;
            logger.info("UNIX identified");
        }
        else {
            os = OS_NT;
            logger.info("NT identified");
        }

        // Enter the user and password strings
        println(user);

        // Wait for password prompt, but watch out for users without password!
        switch (waitForString(PASSWD_PROMPT, true, MENU_PROMPT, true, true)) {
            case 0: logger.info("Got menu instead of password!");
                    handleMenuPrompt();
                    return true;

            case 1: println(passwd);
                    break;
                    // continue as usual

            case 2: handleUnixOrMumpsPrompt();
                    return true;

            case 3: return true; // Got the cav java prompt - ~~~>
        }
        
        logger.info("Password entered");

        // For NT it should immediately get into the cache, or the request for login if failed.
        if (os == OS_NT) {
            switch (waitForString(CAV_JAVA_PROMPT, true, "User authorization failure", true, true)) {
                case 0: return false; // invalid login.

                case 1: return true; // success!

                case 2: logger.info("got the Mumps prompt");
                        println ("D ^MWRAP");
                        waitForString(CAV_JAVA_PROMPT, true);
                        break;

                case 3: return true; // Got the cav java prompt - ~~~>
            }
        }
        
        // For Unix it should get a menu, or the request for login if failed.
        else {// Unix OS
            logger.info("Checking if login succeeded");
            switch (waitForString(MENU_PROMPT, true, "Login incorrect", true, true)) {

                case 0: return false; // invalid login.

                case 1: handleMenuPrompt();
                        return true;

                case 2: handleUnixOrMumpsPrompt();
                        return true;

                case 3: return true; // Got the cav java prompt - ~~~>
            }
        }
        return true;
    }

    /**
     * Got the menu prompt: enter the key and wait for the ~~~> prompt.
     */
    private void handleMenuPrompt() throws IOException{
        logger.info("Got the menu prompt");
        println (MENU_ENTRY_KEY);
        waitForString(CAV_JAVA_PROMPT, true);
    }

    /**
     * Got a prompt of one of the following: % $ # >
     * Since we don't know if it is unix or mumps, we need to test it using the tty command.
     * A typical response from unix will be "/dev/tty..." so we wait for '/'.
     * Mumps doesn't know this command so it will return a syntax error, showing where is
     * the error using the '^' character, therefore this is the break string.
     *
     * After the prompt has been identified, we enter ZCAVJ using mux (unix) or D (mumps).
     * Then all there is to do is sit and wait for the ~~~> prompt.
     */
    private void handleUnixOrMumpsPrompt() throws IOException{
        logger.info("Got some prompt, need to test with tty");
        println ("tty");
        if (waitForString("/", false, "^")) {
            // true: we are in unix prompt
            logger.info("Got the Unix prompt");
            println ("mux ^MWRAP");
        }
        else {
            // false: we are in mumps prompt
            logger.info("Got the Mumps prompt");
            println ("D ^MWRAP");
        }

        // Login passed: get into the cache.
        waitForString(CAV_JAVA_PROMPT, true);
    }

    private void print(char c) throws IOException {
        if (c == '\n') {
            print('\r');
        }
	outputStream.write(c);
	outputStream.flush() ;
	/* NOTE: this doesn't work with PrintWriter, for some reason
	outWriter.print((int)c);
	outWriter.flush();
	if (outWriter.checkError())
	    throw new IOException(TELNET_DISCONNECTED);
	*/
    }

    private void print(String s) throws IOException {
        if (outWriter == null) {
            throw new IOException(TELNET_DISCONNECTED);
        }
        outWriter.print(s);
        if (outWriter.checkError()) {
            throw new IOException(TELNET_DISCONNECTED);
        }
    }

    private void println(String s) throws IOException {
        if (outWriter == null) {
            throw new IOException(TELNET_DISCONNECTED);
        }
        outWriter.println(s);
        if (outWriter.checkError()) {
            throw new IOException(TELNET_DISCONNECTED);
        }
    }

    private boolean waitForString(String str, boolean handleIAC) throws IOException {
        return waitForString(str, handleIAC, "");
    }

    private boolean waitForString(String str, boolean handleIAC, String falseString) throws IOException  {
        return waitForString(str, handleIAC, falseString, false) == 1;
    }

    private int waitForString(String str, boolean handleIAC, String falseString, boolean stopOnUnixPrompt)
	throws IOException {
        return waitForString(str, handleIAC, falseString, stopOnUnixPrompt, false);
	}

    public static final String LICENCE_EXCEEDED_MESSAGE = "LICENSE LIMIT EXCEEDED";
    public static final String USER_EXCEEDED_MESSAGE    = "ABOVE SESSION LIMITS";

    /**
     * Wait for a string from the input stream.
     * @param str                 The main string to wait for and return 1 (true).
     * @param handleIAC           Handle the characters or not.
     * @param falseString         A string that breaks the search and return 0 (false).
     * @param stopOnUnixPrompt    When true, will check for unix prompt: $, %, # or > in the end
     *                            of the stream (positions 1 or 0), and return 2 when got it.
     * @param stopOnCavJavaPrompt When true, will check for the cav java prompt - ~~~>.
     *                            If found, returns 3.
     *
     * @return 1 - success, desired string received.
     *         0 - break string received.
     *         2 - Got some prompt, need to test with tty.
     *         3 - Got the mumps prompt: ~~~>
     */
    private int waitForString(String str, boolean handleIAC, String falseString,
                              boolean stopOnUnixPrompt, boolean stopOnCavJavaPrompt)
	throws IOException {
        char chars[]  = str.toCharArray();
        char licstr[] = LICENCE_EXCEEDED_MESSAGE.toCharArray();
        char usrstr[] = USER_EXCEEDED_MESSAGE.toCharArray();
        char falseStr[] = falseString.toCharArray();
        int  j = 0, k = 0, u = 0;
        int  tildeCount = 0;
        boolean unixPromptReceived = false;

        logger.info("Waiting for string: '" + str + "'");

        // read from the input 'till we have all the strinf at it end
        for (int i = 0; i < chars.length; ) {
            int charCode = inputStream.read();

            // check if end of stream reached (in case of an error)
            if (charCode == -1) {
                logger.severe("Telnet session disconnected from the server");
                throw new IOException(TELNET_DISCONNECTED);
            }

            char c = (char)charCode;

            if (stopOnCavJavaPrompt && c == '>' && tildeCount == 3) {
                return 3;
            }

            if (c == '~') {
                tildeCount++;
            }
            else {
                tildeCount = 0;
            }

            // If we got something else after prompt (which is not a space or null),
            // it is in the middle of text and therefore it is not really a prompt.
            if (c!=' ' && c!=0 && c!=9) {
                unixPromptReceived = false;
            }

            // Check for unix prompt
            if (stopOnUnixPrompt &&
                (c=='$' || c=='%' || c=='#' || c=='>')) {
                unixPromptReceived = true;
            }

            if (handleIAC)
            handleIAC(c);
            // if we've read the next char from the string increment the
            // index in the string, else - go back to beginning
            if (c == chars[i]) {
                i++;
            }
            else {
                i = 0;
            }

            // Check for licence limit string
            if (c == licstr[j]) {
                j++;
                if (j == licstr.length) {
                    throw new IOException(LICENCE_EXCEEDED_MESSAGE);
                }
            }
            else {
                j = 0;
            }

            // Check for USER licence limit string
            if (c == usrstr[u]) {
                u++;
                if (u == usrstr.length) {
                    throw new IOException(USER_EXCEEDED_MESSAGE);
                }
            }
            else {
                u = 0;
            }

            // Check for false return string
            if (falseStr.length > 0 && c == falseStr[k]) {
                k++;
                if (k == falseStr.length) {
                    return 0;
                }
            }
            else {
                k = 0;
            }

            if (unixPromptReceived && inputStream.available() == 0) {
                return 2;
            }
        }

        logger.info("Got it");
        return 1;
    }

    private void handleIAC(char c) throws IOException {
        // check if char is telnet IAC
	if (c == (char)255) {
            char cmd = (char)inputStream.read();
            char opt = (char)inputStream.read();
            switch(opt) {
		//case 1:		// echo
		//	if (cmd == 251) echo = false;
		//	else if (cmd == 252) echo = true;
		//	break;
	    case 3:		// supress go-ahead
		break;
	    case 24:	// terminal type
		if (cmd == 253) {            // IAC WILL terminal-type
		    print((char)255);
		    print((char)251);
		    print((char)24);
		    // IAC SB terminal-type IS <term> IAC SE
		    print((char)255);
		    print((char)250);
		    print((char)24);
		    print((char)0);
		    print(TERM);
		    print((char)255);
		    print((char)240);
		}
		else if (cmd == 250) {
		    while(inputStream.read() != 240)
			;
		}
		break;
	    default:	// some other command
		if (cmd == 253) {    // IAC DONT whatever
		    print((char)255);
		    print((char)252);
		    print(opt);
		}
		break;
	    }
	}
    }

}
