package com.cav.mserver;

import java.io.*;
import java.util.*;
import java.util.logging.Logger;
import java.util.regex.*;

/**
 * A Mumps query that returns a result set.
 * @author Uri Schor
 */
public class MumpsQuery {

	private static Logger logger = Logger.getLogger(MumpsQuery.class
			.getPackage().getName());

	/** The column titles */
	private List columnTitles;

	/** The column types, from MumpsTypes */
	private List columnTypes;

	/** The Mumps command to run */
	private String command;

	/** The Mumps parameters (variables) to set, before running the command */
	private Map parameters;

	/** The result set, each row is a list of columns values */
	private List resultSet;

	/**
	 * Create a new query.
	 * @param command The Mumps command to run
	 * @param parameters The Mumps variable to set before running the command
	 */
	public MumpsQuery(String command, Map parameters) {
		this.command = command;
		this.parameters = parameters;
	}

	/**
	 * Clear the query.
	 */
	public void clear() {
		columnTitles = new ArrayList();
		columnTypes = new ArrayList();
		resultSet = new ArrayList();
	}

	/**
	 * Execute the query.
	 * @param mSession The Mumps session to use.
	 * @throws Exception thrown by {@link org.apache.commons.pool.ObjectPool#borrowObject()}
	 * if there's error in borrowing Mumps session.
	 */
	public void execute(MumpsSession mSession) throws Exception {
		clear();
		InputStream mumpsOutput = mSession.execute(command, parameters);
		BufferedReader mOut = new BufferedReader(new InputStreamReader(
				mumpsOutput, "ISO-8859-8"));
		String line;
		Pattern pattern = Pattern.compile("NAME=(.*),TYPE=(.)");
		while ((line = mOut.readLine()) != null) {
			if (line.startsWith(">>>DATA")) {
				// Data lines (result set) follow
				break;
			}
			// This line holds column description
			Matcher matcher = pattern.matcher(line);
			if (matcher.matches()) {
				String name = matcher.group(1).replace('|', '\n');
				columnTitles.add(name);
				columnTypes.add(new Integer(MumpsTypes
						.get(matcher.group(2))));
			}
			else {
				logger.severe("Bad column definition " + line);
			}
		}
		// Iterate data lines
		int nCols = columnTitles.size();
		while ((line = mOut.readLine()) != null) {
			if (line.trim().length() == 0) {
				break;
			}
			// For each data line, get its tokens, which are separated by '_'
			// characters, into an ArrayList
			int index = 0;
			ArrayList rowList = new ArrayList(nCols);
			boolean endOfTokens = false;
			for (int i = 0; i < nCols; i++) {
				int nextIndex = line.indexOf('_', index);
				if (nextIndex == -1) {
					// No more delimiters - get the rest of the line
					if (!endOfTokens) {
						rowList.add(i, line.substring(index));
						endOfTokens = true;
					} else {
						// pad with empty strings
						rowList.add(i, "");
					}
				}
				else {
					// Get this token, and advance the character index
					rowList.add(i, line.substring(index, nextIndex));
					index = nextIndex + 1;
				}
			}
			resultSet.add(rowList);
		}
	}

	/**
	 * Execute the query, on a session that will be borrowed and released back.
	 * @throws Exception thrown by {@link org.apache.commons.pool.ObjectPool#borrowObject()}
	 * if there's error in borrowing Mumps session.
	 */
	public void execute() throws Exception {
		MumpsSession mSession = (MumpsSession)MumpsSessionPool.instance()
				.borrowObject();
		try {
			execute(mSession);
		}
		finally {
			MumpsSessionPool.instance().returnObject(mSession);
		}
	}

	/**
	 * @return Returns the columnTitles.
	 */
	public List getColumnTitles() {
		return columnTitles;
	}

	/**
	 * @return Returns the columnTypes.
	 */
	public List getColumnTypes() {
		return columnTypes;
	}

	/**
	 * @return Returns the command.
	 */
	public String getCommand() {
		return command;
	}

	/**
	 * @return Returns the parameters.
	 */
	public Map getParameters() {
		return parameters;
	}

	/**
	 * @return Returns the resultSet.
	 */
	public List getResultSet() {
		return resultSet;
	}
}